<?php
namespace App\Jobs;

use App\Events\SendNotification;
use App\Models\Admin\Notifications;
use App\Models\Post;
use App\Models\Setting;
use App\Models\UserFcm;
use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Http\Client\RequestException;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class FetchRssFeedJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $rssFeeds;

    public function __construct(Collection $rssFeeds)
    {
        $this->rssFeeds = $rssFeeds;
    }

    public function handle()
    {
        foreach ($this->rssFeeds as $rssFeed) {
            try {
                $response = Http::get($rssFeed->feed_url);
                if ($response->successful()) {
                    $feedData   = simplexml_load_string($response->body(), 'SimpleXMLElement', LIBXML_NOCDATA);
                    $namespaces = $feedData->getNamespaces(true);

                    // Handle different feed types
                    $items = $this->getFeedItems($feedData);

                    foreach ($items as $item) {
                        $title     = strip_tags(trim((string) $item->title));
                        $slug      = Str::slug($title);
                        $videoData = $this->extractVideoData($item, $namespaces);

                        if ($videoData['video_type'] === 'youtube') {

                            if (! Post::where('title', $title)->orWhere('slug', $slug)->exists()) {
                                $description    = strip_tags(trim((string) $item->description));
                                $link           = (string) $item->link;
                                $pubDate        = trim(strip_tags((string) $item->pubDate));
                                $publishDate    = Carbon::parse($item->pubDate);
                                $newsLangaugeId = $rssFeed->news_language_id;
                                // Extract video information
                                $videoData = $this->extractVideoData($item, $namespaces);

                                DB::enableQueryLog();
                                DB::transaction(function () use ($rssFeed, $title, $slug, $link, $description, $pubDate, $publishDate, $newsLangaugeId, $videoData) {

                                    Post::create([
                                        'channel_id'       => $rssFeed->channel_id,
                                        'topic_id'         => $rssFeed->topic_id,
                                        'title'            => $title,
                                        'resource'         => $link,
                                        'slug'             => $slug,
                                        'image'            => ($videoData['video_type'] === 'youtube') ? $videoData['thumbnail_url'] : $imageUrl, // Use video thumbnail URL or the default image URL
                                        'description'      => $description,
                                        'status'           => 'active',
                                        'pubdate'          => $pubDate,
                                        'publish_date'     => $publishDate,
                                        'news_language_id' => $newsLangaugeId,
                                        // 'image'            => ($videoData['video_type'] === 'youtube') ? $videoData['thumbnail_url'] : null,
                                        'video_thumb'      => ($videoData['video_type'] === 'youtube') ? $videoData['thumbnail_url'] : null,
                                        'video_url'        => $videoData['video_url'],
                                        'video_embed'      => $videoData['video_embed'],
                                        'video_type'       => $videoData['video_type'],
                                        'is_video'         => $videoData['is_video'],
                                        'video'            => $videoData['video_embed'],
                                        'type'             => ! empty($videoData['video_type'])
                                            ? ($videoData['video_type'] === 'youtube' ? 'youtube' : 'video')
                                            : 'post',
                                    ]);
                                });
                            }
                        } else {
                            $slug = Str::slug($title); // generate slug from title

                            if (empty($slug)) {
                                $slug = 'post-' . time();
                            }

                            $originalSlug = $slug;
                            $counter      = 1;
                            while (Post::where('slug', $slug)->exists()) {
                                $slug = $originalSlug . '-' . $counter++;
                            }

                            if (! Post::where('title', $title)->orWhere('slug', $slug)->exists()) {
                                $description    = strip_tags(trim((string) $item->description));
                                $link           = (string) $item->link;
                                $pubDate        = trim(strip_tags((string) $item->pubDate));
                                $imageUrl       = $this->extractImageUrl($item, $namespaces);
                                $publishDate    = Carbon::parse($item->pubDate);
                                $newsLangaugeId = $rssFeed->news_language_id;
                                $defaultImage   = Setting::where('name', 'default_image')->value('value');
                                DB::enableQueryLog();

                                DB::transaction(function () use ($rssFeed, $title, $slug, $link, $description, $pubDate, $publishDate, $imageUrl, $newsLangaugeId, $defaultImage) {

                                    Post::create([
                                        'channel_id'       => $rssFeed->channel_id,
                                        'topic_id'         => $rssFeed->topic_id,
                                        'title'            => $title,
                                        'resource'         => $link,
                                        'slug'             => $slug,
                                        'image'            => $imageUrl ?? url('storage/' . $defaultImage) ?? url('front_end/classic/images/default/post-placeholder.jpg'),
                                        'description'      => $description ?? "",
                                        'type'             => 'post',
                                        'status'           => 'active',
                                        'pubdate'          => $pubDate,
                                        'publish_date'     => $publishDate,
                                        'news_language_id' => $newsLangaugeId,
                                    ]);
                                });
                            }
                        }
                    }
                } else {
                    Log::error("Failed to fetch RSS feed: " . $rssFeed->feed_url);
                }
            } catch (RequestException $e) {
                Log::error("Error fetching RSS feed: " . $rssFeed->feed_url . " - " . $e->getMessage());
            } catch (\Throwable $e) {
                Log::error("Error processing RSS feed: " . $rssFeed->feed_url . " - " . $e->getMessage());
            }
        }

        $existPostSlugs = Notifications::select('slug')->pluck('slug')->toArray();
        $post           = Post::select('id', 'title', 'description', 'image', 'slug')->whereNotIn('slug', $existPostSlugs)->orderBy('publish_date', 'desc')->first();

        $fcmIds = UserFcm::select(['fcm_id', 'platform'])->get()->toArray();

        /* Call an event for send notification */
        event(new SendNotification($post->title, $post->description, $post->image, $post->slug, $fcmIds));
    }

    /**
     * Get feed items based on feed type (RSS or Atom)
     */
    private function getFeedItems($feedData)
    {
        // Check if it's an Atom feed
        if (isset($feedData->entry)) {
            return $feedData->entry;
        }

        // Default to RSS format
        return $feedData->channel->item ?? [];
    }

    /**
     * Extract publication date from different feed formats
     */
    private function extractPubDate($item)
    {
        // For Atom feeds
        if (isset($item->published)) {
            return trim(strip_tags((string) $item->published));
        }

        // For RSS feeds
        if (isset($item->pubDate)) {
            return trim(strip_tags((string) $item->pubDate));
        }

        // Fallback to current date
        return Carbon::now()->toDateTimeString();
    }

    private function extractVideoData($item, $namespaces)
    {
        $videoData = [
            'video_url'     => null,
            'video_embed'   => null,
            'video_type'    => null,
            'is_video'      => false,
            'thumbnail_url' => null, // Add thumbnail support
        ];

        // Check for YouTube video ID in namespaces
        foreach ($namespaces as $prefix => $namespace) {
            if ($prefix === 'yt' && isset($item->children($namespace)->videoId)) {
                $videoId                    = (string) $item->children($namespace)->videoId;
                $videoData['video_url']     = "https://www.youtube.com/watch?v=" . $videoId;
                $videoData['video_embed']   = "https://www.youtube.com/embed/" . $videoId;
                $videoData['video_type']    = 'youtube';
                $videoData['is_video']      = true;
                $videoData['thumbnail_url'] = "https://img.youtube.com/vi/" . $videoId . "/maxresdefault.jpg";
                break;
            }
        }

        // Check for media:content with video type
        if (isset($item->children('media', true)->content)) {
            foreach ($item->children('media', true)->content as $mediaContent) {
                $type = (string) $mediaContent->attributes()->type;
                if (strpos($type, 'video') !== false || $type === 'application/x-shockwave-flash') {
                    $url = (string) $mediaContent->attributes()->url;

                    // Extract video ID from YouTube URLs
                    if (preg_match('/youtube\.com\/v\/([a-zA-Z0-9_-]+)/', $url, $matches)) {
                        $videoId                    = $matches[1];
                        $videoData['video_url']     = "https://www.youtube.com/watch?v=" . $videoId;
                        $videoData['video_embed']   = "https://www.youtube.com/embed/" . $videoId;
                        $videoData['video_type']    = 'youtube';
                        $videoData['is_video']      = true;
                        $videoData['thumbnail_url'] = "https://img.youtube.com/vi/" . $videoId . "/maxresdefault.jpg";
                    } else {
                        $videoData['video_url']   = $url;
                        $videoData['video_embed'] = $url;
                        $videoData['video_type']  = $this->detectVideoType($url);
                        $videoData['is_video']    = true;
                    }
                    break;
                }
            }
        }

        // Check for media:thumbnail (most common way thumbnails are provided)
        if (isset($item->children('media', true)->thumbnail)) {
            $thumbnails = $item->children('media', true)->thumbnail;

            // If multiple thumbnails, get the largest one
            $bestThumbnail = null;
            $maxWidth      = 0;

            foreach ($thumbnails as $thumbnail) {
                $width  = (int) $thumbnail->attributes()->width;
                $height = (int) $thumbnail->attributes()->height;
                $url    = (string) $thumbnail->attributes()->url;

                if ($width > $maxWidth || ! $bestThumbnail) {
                    $maxWidth      = $width;
                    $bestThumbnail = $url;
                }
            }

            if ($bestThumbnail) {
                $videoData['thumbnail_url'] = $bestThumbnail;
            }
        }

        // Check for media:group containing thumbnails
        if (isset($item->children('media', true)->group)) {
            $mediaGroup = $item->children('media', true)->group;
            if (isset($mediaGroup->thumbnail)) {
                $thumbnails = $mediaGroup->thumbnail;

                $bestThumbnail = null;
                $maxWidth      = 0;

                foreach ($thumbnails as $thumbnail) {
                    $width = (int) $thumbnail->attributes()->width;
                    $url   = (string) $thumbnail->attributes()->url;

                    if ($width > $maxWidth || ! $bestThumbnail) {
                        $maxWidth      = $width;
                        $bestThumbnail = $url;
                    }
                }

                if ($bestThumbnail) {
                    $videoData['thumbnail_url'] = $bestThumbnail;
                }
            }
        }

        // Check for enclosure with video type
        if (isset($item->enclosure)) {
            $type = (string) $item->enclosure['type'];
            if (strpos($type, 'video') !== false) {
                $url                      = (string) $item->enclosure['url'];
                $videoData['video_url']   = $url;
                $videoData['video_embed'] = $url;
                $videoData['video_type']  = $this->detectVideoType($url);
                $videoData['is_video']    = true;
            }
        }

        // Check link for video URLs
        if (! $videoData['is_video']) {
            $link = (string) $item->link;
            if ($this->isVideoUrl($link)) {
                $videoData['video_url']  = $link;
                $videoData['video_type'] = $this->detectVideoType($link);
                $videoData['is_video']   = true;

                // Generate embed URL for supported platforms
                $embedUrl = $this->generateEmbedUrl($link, $videoData['video_type']);
                if ($embedUrl) {
                    $videoData['video_embed'] = $embedUrl;
                }

                // Generate thumbnail for YouTube links
                if ($videoData['video_type'] === 'youtube' && ! $videoData['thumbnail_url']) {
                    $videoData['thumbnail_url'] = $this->getYouTubeThumbnail($link);
                }
            }
        }

        // Fallback: Check for any image in media:content as potential thumbnail
        if (! $videoData['thumbnail_url'] && isset($item->children('media', true)->content)) {
            foreach ($item->children('media', true)->content as $mediaContent) {
                $type = (string) $mediaContent->attributes()->type;
                if (strpos($type, 'image') !== false) {
                    $videoData['thumbnail_url'] = (string) $mediaContent->attributes()->url;
                    break;
                }
            }
        }

        return $videoData;
    }

// Helper method to extract YouTube thumbnail from URL
    private function getYouTubeThumbnail($url)
    {
        // Extract video ID from various YouTube URL formats
        $patterns = [
            '/youtube\.com\/watch\?v=([a-zA-Z0-9_-]+)/',
            '/youtube\.com\/embed\/([a-zA-Z0-9_-]+)/',
            '/youtu\.be\/([a-zA-Z0-9_-]+)/',
            '/youtube\.com\/v\/([a-zA-Z0-9_-]+)/',
        ];

        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $url, $matches)) {
                return "https://img.youtube.com/vi/" . $matches[1] . "/maxresdefault.jpg";
            }
        }

        return null;
    }

    /**
     * Check if URL is a video URL
     */
    private function isVideoUrl($url)
    {
        $videoPatterns = [
            'youtube.com',
            'youtu.be',
            'vimeo.com',
            'dailymotion.com',
            'twitch.tv',
            'facebook.com/watch',
            'instagram.com/p/',
            'tiktok.com',
        ];

        foreach ($videoPatterns as $pattern) {
            if (strpos($url, $pattern) !== false) {
                return true;
            }
        }

        return false;
    }

    /**
     * Detect video type from URL
     */
    private function detectVideoType($url)
    {
        if (strpos($url, 'youtube.com') !== false || strpos($url, 'youtu.be') !== false) {
            return 'youtube';
        } elseif (strpos($url, 'vimeo.com') !== false) {
            return 'vimeo';
        } elseif (strpos($url, 'dailymotion.com') !== false) {
            return 'dailymotion';
        } elseif (strpos($url, 'twitch.tv') !== false) {
            return 'twitch';
        } elseif (strpos($url, 'facebook.com') !== false) {
            return 'facebook';
        } elseif (strpos($url, 'instagram.com') !== false) {
            return 'instagram';
        } elseif (strpos($url, 'tiktok.com') !== false) {
            return 'tiktok';
        }

        return 'other';
    }

    /**
     * Generate embed URL for supported video platforms
     */
    private function generateEmbedUrl($url, $type)
    {
        switch ($type) {
            case 'youtube':
                // Extract video ID from various YouTube URL formats
                preg_match('/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/', $url, $matches);
                if (isset($matches[1])) {
                    return "https://www.youtube.com/embed/" . $matches[1];
                }
                break;

            case 'vimeo':
                preg_match('/vimeo\.com\/(\d+)/', $url, $matches);
                if (isset($matches[1])) {
                    return "https://player.vimeo.com/video/" . $matches[1];
                }
                break;

            case 'dailymotion':
                preg_match('/dailymotion\.com\/video\/([a-zA-Z0-9]+)/', $url, $matches);
                if (isset($matches[1])) {
                    return "https://www.dailymotion.com/embed/video/" . $matches[1];
                }
                break;
        }

        return null;
    }

    private function extractImageUrl($item, $namespaces)
    {
        // Check for thumbnail in namespaces
        foreach ($namespaces as $prefix => $namespace) {
            if (isset($item->children($namespace)->thumbnail)) {
                $data = (string) $item->children($namespace)->thumbnail->attributes()->url;
            }
        }

        // Check for enclosure URL
        if (isset($item->enclosure['url'])) {
            $data = (string) $item->enclosure['url'];
        }

        // Check for media:content URL
        if (isset($item->children('media', true)->content)) {
            foreach ($item->children('media', true)->content as $mediaContent) {
                if ((string) $mediaContent->attributes()->type === 'image/jpeg') {
                    $data = (string) $mediaContent->attributes()->url;
                }
            }
        }

        // Check for img tag in description
        if (isset($item->description)) {
            $matches = [];
            preg_match('/<img[^>]+src="([^">]+)"/', (string) $item->description, $matches);
            if (! empty($matches[1])) {
                return $matches[1];
            }
        }

        return $data ?? null;
    }
}
